'use client';

import { useState, useEffect, useRef, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import Link from 'next/link';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import NavBar from '@/components/NavBar';

interface RecordData {
  [key: string]: any;
}

// Main content component that uses useSearchParams
function ClientIdContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    currentTable,
    setCurrentHostId,
    setCurrentDatabaseName,
    setCurrentTableName
  } = useDatabaseContext();
  
  const [record, setRecord] = useState<RecordData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  
  // Get client ID from URL
  const recordId = searchParams ? searchParams.get('clid') : null;
  
  // Set the table to Clients for this specialized view
  useEffect(() => {
    // Default to Clients table for client pages
    setCurrentTableName('Clients');
    
    // Log for debugging
    console.log('Client page: Setting default table and using record ID:', { 
      recordId,
      currentHostId, 
      currentDatabaseName 
    });
  }, [setCurrentTableName, recordId, currentHostId, currentDatabaseName]);

  // Use a ref to track if we've already successfully loaded a record
  // This helps prevent errors during React's strict mode double-mounting
  const recordLoadedRef = useRef(false);
  
  // Fetch record data when component mounts or when parameters change
  useEffect(() => {
    const fetchRecord = async () => {
      if (!currentHostId || !currentDatabaseName || !currentTableName) {
        setError('Please select a host, database, and table');
        setLoading(false);
        return;
      }
      
      if (!recordId) {
        setError('No record ID provided');
        setLoading(false);
        return;
      }
      
      try {
        setLoading(true);
        setError(null);
        
        // Use the correct OData URL format with parentheses for ID
        // The correct format is: /{tableName}({id}) not /{tableName}/{id}
        
        // This is the format that works in Postman:
        // https://db-data.net/fmi/odata/v4/TrackPad-FMS-test/Clients(29)
        const url = `/api/odata/${currentDatabaseName}/${currentTableName}(${recordId})`;
        
        // Log the full URL path for debugging
        console.log(`Using direct ID fetch approach from reference code`);
        console.log(`Full request URL: ${url}`);
        console.log(`Full OData URL would be: https://db-data.net/fmi/odata/v4/${currentDatabaseName}/${currentTableName}(${recordId})`);
        console.log(`Current database: ${currentDatabaseName}`);
        console.log(`Current table: ${currentTableName}`);
        console.log(`Record ID: ${recordId}`);
        
        const response = await fetch(url, {
          headers: {
            'X-Host-Id': currentHostId,
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'OData-Version': '4.0',
            'OData-MaxVersion': '4.0'
          },
          cache: 'no-store'
        });
        
        if (!response.ok) {
          const errorData = await response.json().catch(() => ({}));
          throw new Error(errorData.message || `Failed to fetch record: ${response.statusText}`);
        }
        
        const data = await response.json();
        
        // Log the response for debugging
        console.log('Response data:', JSON.stringify(data, null, 2));
        
        // Handle OData response format (could be direct object or have a 'value' property)
        // This matches the reference code approach
        if (data) {
          console.log('Record found, setting state');
          setRecord(data.value || data);
          recordLoadedRef.current = true;
          return; // Exit early on success to prevent further processing
        } 
        
        // Only throw an error if we haven't already successfully loaded a record
        // This prevents errors during React's double-mounting in development
        if (!recordLoadedRef.current) {
          console.log('No record found in the response');
          throw new Error(`No client found with ID=${recordId}`);
        } else {
          console.log('Ignoring null response since record was already loaded');
        }
      } catch (err) {
        console.error('Error fetching record:', err);
        setError(err instanceof Error ? err.message : 'An unknown error occurred');
      } finally {
        setLoading(false);
      }
    };

    fetchRecord();
  }, [recordId, currentHostId, currentDatabaseName, currentTableName]);

  // Filter and order fields based on table configuration
  const getDisplayFields = (record: RecordData): [string, any][] => {
    if (!record) return [];
    
    const tableFields = currentTable?.fields || [];
    const preferredOrder = currentTable?.preferredColumnOrder || [];
    
    // Get all record keys excluding OData metadata
    const recordKeys = Object.keys(record).filter(key => !key.startsWith('@odata'));
    
    // Start with preferred order fields that exist in the record
    const orderedKeys = [
      // First include preferred columns in order
      ...preferredOrder.filter(col => recordKeys.includes(col)),
      // Then include any remaining fields from the config that exist in the record
      ...tableFields
        .filter(field => !field.hidden && !preferredOrder.includes(field.fieldName))
        .map(field => field.fieldName)
        .filter(fieldName => recordKeys.includes(fieldName)),
      // Finally include any fields from the record that aren't in the config
      ...recordKeys.filter(key => 
        !preferredOrder.includes(key) && 
        !tableFields.some(field => field.fieldName === key)
      ).sort()
    ];
    
    // Map keys to [key, value] pairs
    return orderedKeys.map(key => [key, record[key]]);
  };

  // Format field name for display (e.g., 'firstName' -> 'First Name')
  const formatFieldName = (field: string): string => {
    // Check if we have a display name in the table configuration
    const fieldConfig = currentTable?.fields?.find(f => f.fieldName === field);
    if (fieldConfig?.displayName) {
      return fieldConfig.displayName;
    }
    
    // Otherwise, format the field name
    return field
      .replace(/([A-Z])/g, ' $1')
      .replace(/^./, str => str.toUpperCase())
      .trim();
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="bg-white rounded-lg shadow-lg p-6">
            <div className="animate-pulse flex space-x-4">
              <div className="flex-1 space-y-6 py-1">
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                <div className="space-y-3">
                  <div className="grid grid-cols-3 gap-4">
                    <div className="h-4 bg-gray-200 rounded col-span-2"></div>
                    <div className="h-4 bg-gray-200 rounded col-span-1"></div>
                  </div>
                  <div className="h-4 bg-gray-200 rounded"></div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  // Format cell value for display based on field type
  const formatCellValue = (field: string, value: any): React.ReactNode => {
    if (value === null || value === undefined) {
      return <span className="text-gray-400">-</span>;
    }
    
    // Get field configuration if available
    const fieldConfig = currentTable?.fields?.find(f => f.fieldName === field);
    
    if (fieldConfig) {
      // Format based on field type
      switch (fieldConfig.type) {
        case 'date':
          try {
            const date = new Date(value);
            return date.toLocaleDateString();
          } catch (e) {
            return String(value);
          }
        case 'boolean':
          return value ? 'Yes' : 'No';
        case 'number':
          return typeof value === 'number' ? value.toLocaleString() : String(value);
        default:
          if (typeof value === 'object') {
            return <pre className="text-xs">{JSON.stringify(value, null, 2)}</pre>;
          }
          return String(value);
      }
    }
    
    // Default formatting if no field config
    if (typeof value === 'object') {
      return <pre className="text-xs">{JSON.stringify(value, null, 2)}</pre>;
    }
    return String(value);
  };

  if (error) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="bg-white rounded-lg shadow-lg p-6">
            <div className="bg-red-50 border-l-4 border-red-400 p-4">
              <div className="flex">
                <div className="flex-shrink-0">
                  <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                    <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clipRule="evenodd" />
                  </svg>
                </div>
                <div className="ml-3">
                  <p className="text-sm text-red-700">{error}</p>
                  <div className="mt-4">
                    <Link href="/list" className="text-sm font-medium text-red-700 hover:text-red-600">
                      &larr; Back to list
                    </Link>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-100">
      <NavBar />
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="px-3 sm:px-6 py-4 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
              <h1 className="text-xl font-semibold text-gray-900">
                Record Details
              </h1>
              <div className="flex gap-2">
                <Link
                  href="/list"
                  className="px-3 py-1 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-sm"
                >
                  Back to List
                </Link>
                <Link
                  href={`/client-update?id=${recordId}`}
                  className="px-3 py-1 bg-blue-600 text-white rounded hover:bg-blue-700 text-sm"
                >
                  Edit
                </Link>
              </div>
            </div>
          </div>
          
          <div className="p-1 sm:p-3 md:p-4">
            {/* Database selector removed - using default Clients table */}
            
            {record && (
              <div className="mt-6 overflow-x-auto">
                <div className="inline-block min-w-full align-middle">
                  <div className="overflow-hidden shadow ring-1 ring-black ring-opacity-5 sm:rounded-lg">
                    <table className="min-w-full divide-y divide-gray-300">
                      <thead className="bg-gray-50">
                        <tr>
                          <th scope="col" className="py-3.5 pl-4 pr-3 text-left text-sm font-semibold text-gray-900 sm:pl-6">Field</th>
                          <th scope="col" className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900">Value</th>
                        </tr>
                      </thead>
                      <tbody className="divide-y divide-gray-200 bg-white">
                        {getDisplayFields(record).map(([key, value]) => (
                          <tr key={key}>
                            <td className="whitespace-nowrap py-4 pl-4 pr-3 text-sm font-medium text-gray-900 sm:pl-6">
                              {formatFieldName(key)}
                            </td>
                            <td className="whitespace-pre-wrap px-3 py-4 text-sm text-gray-500 break-words">
                              {formatCellValue(key, value)}
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

// Export the main component wrapped in Suspense
export default function SingleRecordPage() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="bg-white shadow rounded-lg p-6">
              <h1 className="text-2xl font-bold mb-6">Client Details</h1>
              <p>Loading client details...</p>
            </div>
          </div>
        </div>
      </div>
    }>
      <ClientIdContent />
    </Suspense>
  );
}
