'use client';

import { useState, useEffect, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import Link from 'next/link';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import NavBar from '@/components/NavBar';

interface RecordData {
  [key: string]: any;
}

interface FormField {
  key: string;
  value: any;
  type: string;
  required: boolean;
}

// Main content component that uses useSearchParams
function UpdateRecordContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    currentTable,
    setCurrentHostId,
    setCurrentDatabaseName,
    setCurrentTableName
  } = useDatabaseContext();
  
  const [record, setRecord] = useState<RecordData | null>(null);
  const [formFields, setFormFields] = useState<FormField[]>([]);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [actualRecordId, setActualRecordId] = useState<string | null>(null);
  const [saveSuccess, setSaveSuccess] = useState(false);
  
  // State for client name and ID from session storage
  const [clientName, setClientName] = useState<string>('');
  const [storedClientId, setStoredClientId] = useState<string>('');
  
  // Load client name and ID from session storage on component mount
  useEffect(() => {
    // Need to check if window is defined since this is a Next.js app with SSR
    if (typeof window !== 'undefined') {
      const storedName = sessionStorage.getItem('cl-name') || '';
      const storedId = sessionStorage.getItem('clid') || '';
      
      setClientName(storedName);
      setStoredClientId(storedId);
      
      console.log('Retrieved from session storage:', {
        'cl-name': storedName,
        'clid': storedId
      });
    }
  }, []);
  
  // Get parameters from URL
  const recordId = searchParams ? searchParams.get('clid') : null;
  const hostId = searchParams ? searchParams.get('host') : null;
  const databaseName = searchParams ? searchParams.get('database') : null;
  const tableName = searchParams ? searchParams.get('table') : null;
  
  // Set the table to Clients for this specialized view
  useEffect(() => {
    // Default to Clients table for client pages
    setCurrentTableName('Clients');
    
    // Log for debugging
    console.log('Client update page: Setting default table and using record ID:', { 
      recordId,
      currentHostId, 
      currentDatabaseName 
    });
  }, [setCurrentTableName, recordId, currentHostId, currentDatabaseName]);
  
  // Set the context values from URL parameters if provided
  useEffect(() => {
    if (hostId) setCurrentHostId(hostId);
    if (databaseName) setCurrentDatabaseName(databaseName);
    if (tableName) setCurrentTableName(tableName);
  }, [hostId, databaseName, tableName, setCurrentHostId, setCurrentDatabaseName, setCurrentTableName]);

  // Fetch record data when component mounts or when parameters change
  useEffect(() => {
    const fetchRecord = async () => {
      if (!currentHostId || !currentDatabaseName || !currentTableName) {
        setError('Please select a host, database, and table');
        setLoading(false);
        return;
      }
      
      // If no record ID, we're creating a new record
      if (!recordId) {
        setRecord({});
        convertRecordToFormFields({});
        setLoading(false);
        return;
      }

      try {
        setLoading(true);
        setError(null);
        setSaveSuccess(false);
        
        // Using OData filter query with Name_id eq clid
        console.log('Using OData filter query with Name_id eq clid');
        const encodedDbName = encodeURIComponent(currentDatabaseName);
        const encodedTableName = encodeURIComponent(currentTableName);
        const url = `/api/odata/${encodedDbName}/${encodedTableName}?$filter=Name_id eq ${recordId}`;
        console.log(`Full request URL: ${url}`);
        console.log(`Current database: ${currentDatabaseName}`);
        console.log(`Current table: ${currentTableName}`);
        console.log(`Record ID: ${recordId}`);
        
        const response = await fetch(url, {
          headers: {
            'X-Host-Id': currentHostId,
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'OData-Version': '4.0',
            'OData-MaxVersion': '4.0'
          },
          cache: 'no-store' // Prevent caching of the response
        });
        
        if (!response.ok) {
          throw new Error(`Failed to fetch record: ${response.statusText}`);
        }
        
        const data = await response.json();
        console.log('Response data:', JSON.stringify(data, null, 2));
        
        // Handle OData response format with nested data array structure
        let recordData = null;
        
        if (data.data && Array.isArray(data.data) && data.data.length > 0) {
          recordData = data.data[0];
          console.log('Record found in data array, setting state');
        } else if (data.value && Array.isArray(data.value) && data.value.length > 0) {
          recordData = data.value[0];
          console.log('Record found in value array, setting state');
        } else if (Array.isArray(data) && data.length > 0) {
          recordData = data[0];
          console.log('Record found in direct array, setting state');
        } else {
          recordData = data;
          console.log('Using direct data object');
        }
        
        if (!recordData || typeof recordData !== 'object' || Object.keys(recordData).length === 0) {
          throw new Error(`No client found with Name_id=${recordId}`);
        }
        
        // Extract actual record ID from OData response
        let odataId = null;
        if (recordData['@odata.id']) {
          const match = recordData['@odata.id'].match(/\((\d+)\)/);
          if (match && match[1]) {
            odataId = match[1];
            console.log('Extracted actual record ID from @odata.id:', odataId);
          }
        } else if (recordData['_recid']) {
          odataId = recordData['_recid'];
          console.log('Using _recid as actual record ID:', odataId);
        }
        
        setActualRecordId(odataId);
        setRecord(recordData);
        convertRecordToFormFields(recordData);
      } catch (err) {
        console.error('Error fetching record:', err);
        setError(err instanceof Error ? err.message : 'An unknown error occurred');
      } finally {
        setLoading(false);
      }
    };

    fetchRecord();
  }, [recordId, currentHostId, currentDatabaseName, currentTableName]);

  // Get editable and read-only fields from table configuration or use defaults
  const editableFields = currentTable?.editableFields || ['Name', 'Email', 'Phone', 'Company', 'Notes', 'Status'];
  const readOnlyFields = currentTable?.readOnlyFields || ['z_recid', 'Name_id'];
  
  // Use displayShort fields from the current table configuration
  const displayFields = currentTable?.displayShort || [
    'z_recid',
    'Name',
    'Email',
    'Phone',
    'Company',
    'Notes',
    'Status'
  ];
  
  // Fields that should be completely hidden
  // Combine system hidden fields with configured hidden fields
  const systemHiddenFields = ['@id', '@edit Link', '@context', '_recid'];
  const configuredHiddenFields = currentTable?.hiddenFields || [];
  const hiddenFields = [...systemHiddenFields, ...configuredHiddenFields];
  
  // Convert record data to form fields
  const convertRecordToFormFields = (data: RecordData) => {
    if (!data) {
      setFormFields([]);
      return;
    }
    
    // Get all keys from the record, excluding OData metadata fields and hidden fields
    // Only include fields that are in the displayShort configuration
    const keys = Object.keys(data).filter(key => 
      !key.startsWith('@odata') && 
      !hiddenFields.includes(key) &&
      displayFields.includes(key)
    );
    
    // Get field metadata from current table configuration if available
    const fieldConfigs = currentTable?.fields || [];
    
    const fields: FormField[] = [];
    
    // Add fields from record
    keys.forEach(key => {
      // Find field config if available
      const fieldConfig = fieldConfigs.find(f => f.fieldName === key);
      
      // Determine if this field should be editable
      const isEditable = editableFields.includes(key);
      
      fields.push({
        key,
        value: data[key] === null ? '' : data[key],
        type: isEditable ? getFieldType(key, data[key], fieldConfig?.type) : 'readonly',
        required: isEditable && (fieldConfig?.required || false)
      });
    });
    
    // Add any configured editable fields that weren't in the record
    fieldConfigs.forEach(config => {
      if (editableFields.includes(config.fieldName) && 
          !hiddenFields.includes(config.fieldName) && 
          !fields.some(f => f.key === config.fieldName)) {
        fields.push({
          key: config.fieldName,
          value: '',
          type: getFieldType(config.fieldName, '', config.type),
          required: config.required || false
        });
      }
    });
    
    // Sort fields based on the displayShort order
    fields.sort((a, b) => {
      // First check if both fields are in displayShort
      const aIndex = displayFields.indexOf(a.key);
      const bIndex = displayFields.indexOf(b.key);
      
      // If both are in displayShort, sort by their order there
      if (aIndex !== -1 && bIndex !== -1) {
        return aIndex - bIndex;
      }
      
      // If only one field is in displayShort, prioritize it
      if (aIndex !== -1) return -1;
      if (bIndex !== -1) return 1;
      
      // Otherwise sort by editability
      const aIsEditable = editableFields.includes(a.key);
      const bIsEditable = editableFields.includes(b.key);
      
      if (aIsEditable && !bIsEditable) return -1;
      if (!aIsEditable && bIsEditable) return 1;
      
      // If both are editable, sort alphabetically
      return a.key.localeCompare(b.key);
    });
    
    setFormFields(fields);
  };

  // Determine field type based on key name and value
  const getFieldType = (key: string, value: any, configType?: string): string => {
    // Use configuration type if available
    if (configType) return configType;
    
    // Otherwise infer from key name and value
    const keyLower = key.toLowerCase();
    
    if (keyLower.includes('date')) return 'date';
    if (keyLower.includes('time')) return 'time';
    if (keyLower.includes('email')) return 'email';
    if (keyLower.includes('phone')) return 'tel';
    if (keyLower.includes('url')) return 'url';
    if (keyLower.includes('password')) return 'password';
    
    if (typeof value === 'number') return 'number';
    if (typeof value === 'boolean') return 'checkbox';
    
    if (keyLower === 'status' || keyLower === 'pri' || keyLower === 'priority') return 'select';
    
    return 'text';
  };

  // Format field name for display (e.g., 'firstName' -> 'First Name')
  const formatFieldName = (field: string): string => {
    return field
      .replace(/([A-Z])/g, ' $1')
      .replace(/^./, str => str.toUpperCase())
      .trim();
  };

  // Handle form field changes
  const handleFieldChange = (index: number, value: any) => {
    const updatedFields = [...formFields];
    updatedFields[index].value = value;
    setFormFields(updatedFields);
  };

  // Handle form submission
  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);
    setError('');
    setSaveSuccess(false);
    
    try {
      // Convert form fields to record object - only include editable fields
      const updatedRecord: Record<string, any> = {};
      formFields.forEach(field => {
        // Only include editable fields in the update payload
        if (editableFields.includes(field.key)) {
          updatedRecord[field.key] = field.value;
        }
      });
      
      // Extra safety check - remove any OData metadata fields
      Object.keys(updatedRecord).forEach(key => {
        if (key.startsWith('@') || key === 'context') {
          delete updatedRecord[key];
        }
      });
      
      const encodedDbName = encodeURIComponent(currentDatabaseName);
      const encodedTableName = encodeURIComponent(currentTableName);
      let response;
      let method;
      let url;
      
      if (actualRecordId) {
        // Update existing record using actual record ID
        method = 'PATCH';
        url = `/api/odata/${encodedDbName}/${encodedTableName}/${actualRecordId}`;
        console.log('Update URL with actual record ID:', url);
      } else if (recordId) {
        // Fallback to using Name_id if actual record ID is not available
        method = 'PATCH';
        url = `/api/odata/${encodedDbName}/${encodedTableName}/${recordId}`;
        console.log('Update URL with Name_id as fallback:', url);
      } else {
        // Create new record
        method = 'POST';
        url = `/api/odata/${encodedDbName}/${encodedTableName}`;
      }
      
      const headers: Record<string, string> = {
        'X-Host-Id': currentHostId,
        'Accept': 'application/json',
        'Content-Type': 'application/json',
        'OData-Version': '4.0',
        'OData-MaxVersion': '4.0'
      };
      
      // Add Authorization header if needed (handled by API route)
      
      response = await fetch(url, {
        method,
        headers,
        body: JSON.stringify(updatedRecord),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || `Error ${response.status}: ${response.statusText}`);
      }
      
      const result = await response.json();
      setSaveSuccess(true);
      
      // Redirect to the client page after successful save
      setTimeout(() => {
        const newRecordId = recordId || (result && result._recid);
        if (newRecordId) {
          // Redirect to client page with clid parameter
          console.log('Redirecting to client page with clid:', newRecordId);
          router.push(`/client?clid=${newRecordId}`);
        } else {
          router.push('/list');
        }
      }, 1500);
      
    } catch (err) {
      console.error('Error saving record:', err);
      setError(err instanceof Error ? err.message : 'An unknown error occurred');
    } finally {
      setSaving(false);
    }
  };

  // Get options for select fields
  const getSelectOptions = (fieldKey: string): string[] => {
    const keyLower = fieldKey.toLowerCase();
    
    if (keyLower === 'status') {
      return ['New', 'In Progress', 'Completed', 'On Hold', 'Cancelled'];
    }
    
    if (keyLower === 'pri' || keyLower === 'priority') {
      return ['1', '2', '3', '4', '5'];
    }
    
    // Check if field has options in table configuration
    const fieldConfig = currentTable?.fields.find(f => f.fieldName === fieldKey);
    if (fieldConfig?.options) {
      return fieldConfig.options;
    }
    
    return [];
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="bg-white rounded-lg shadow-lg p-6">
            <div className="animate-pulse flex space-x-4">
              <div className="flex-1 space-y-6 py-1">
                <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                <div className="space-y-3">
                  <div className="grid grid-cols-3 gap-4">
                    <div className="h-4 bg-gray-200 rounded col-span-2"></div>
                    <div className="h-4 bg-gray-200 rounded col-span-1"></div>
                  </div>
                  <div className="h-4 bg-gray-200 rounded"></div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-100">
      <NavBar />
      
      {/* Client information display from session storage */}
      {(clientName || storedClientId) && (
        <div className="max-w-7xl mx-auto pt-4 pb-0 sm:px-6 lg:px-8">
          <div className="bg-blue-50 rounded-lg shadow-md p-4 mb-4">
            <div className="text-center">
              <h2 className="text-2xl font-bold text-blue-800">
                Client: {clientName || 'Unknown'}
              </h2>
              <p className="text-lg text-blue-600">
                Client ID: {storedClientId || recordId || 'Not specified'}
              </p>
            </div>
          </div>
        </div>
      )}
      
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="px-3 sm:px-6 py-4 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
              <h1 className="text-xl font-semibold text-gray-900">
                {recordId ? 'Update Record' : 'Create New Record'}
              </h1>
              <div className="flex gap-2">
                <Link
                  href={`/client${recordId ? `?clid=${recordId}` : ''}`}
                  className="px-3 py-1 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-sm"
                >
                  Cancel
                </Link>
              </div>
            </div>
          </div>
          
          <div className="p-1 sm:p-3 md:p-4 database-content">
            {error && (
              <div className="bg-red-50 border-l-4 border-red-400 p-4 my-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-red-700">{error}</p>
                  </div>
                </div>
              </div>
            )}
            
            {saveSuccess && (
              <div className="bg-green-50 border-l-4 border-green-400 p-4 my-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-green-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-green-700">Record saved successfully! Redirecting...</p>
                  </div>
                </div>
              </div>
            )}
            
            <form onSubmit={handleSubmit} className="mt-6 space-y-6">
              {/* Editable Fields Section */}
              <div className="mb-8">
                <h2 className="text-lg font-medium text-gray-900 mb-4 pb-2 border-b border-gray-200">Editable Fields</h2>
                <div className="space-y-6">
                  {formFields
                    .filter(field => editableFields.includes(field.key))
                    .map((field, index) => {
                      const originalIndex = formFields.findIndex(f => f.key === field.key);
                      return (
                        <div key={field.key} className="grid grid-cols-1 gap-2">
                          <label htmlFor={field.key} className="block text-sm font-medium text-gray-700">
                            {formatFieldName(field.key)}
                            {field.required && <span className="text-red-500 ml-1">*</span>}
                          </label>
                          
                          {field.type === 'select' ? (
                            <select
                              id={field.key}
                              value={field.value}
                              onChange={(e) => handleFieldChange(originalIndex, e.target.value)}
                              className="block w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                              required={field.required}
                            >
                              <option value="">-- Select --</option>
                              {getSelectOptions(field.key).map(option => (
                                <option key={option} value={option}>{option}</option>
                              ))}
                            </select>
                          ) : field.type === 'textarea' ? (
                            <textarea
                              id={field.key}
                              value={field.value}
                              onChange={(e) => handleFieldChange(originalIndex, e.target.value)}
                              rows={4}
                              className="block w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                              required={field.required}
                            />
                          ) : field.type === 'checkbox' ? (
                            <div className="flex items-center">
                              <input
                                id={field.key}
                                type="checkbox"
                                checked={field.value}
                                onChange={(e) => handleFieldChange(originalIndex, e.target.checked)}
                                className="h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                              />
                              <label htmlFor={field.key} className="ml-2 block text-sm text-gray-900">
                                {field.value ? 'Yes' : 'No'}
                              </label>
                            </div>
                          ) : (
                            <input
                              id={field.key}
                              type={field.type}
                              value={field.value}
                              onChange={(e) => handleFieldChange(originalIndex, e.target.value)}
                              className="block w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                              required={field.required}
                            />
                          )}
                        </div>
                      );
                    })
                  }
                </div>
              </div>
              
              {/* Read-Only Fields Section */}
              <div>
                <h2 className="text-lg font-medium text-gray-900 mb-4 pb-2 border-b border-gray-200">Read-Only Fields</h2>
                <div className="space-y-6">
                  {formFields
                    .filter(field => !editableFields.includes(field.key) && !hiddenFields.includes(field.key))
                    .map((field) => (
                      <div key={field.key} className="grid grid-cols-1 gap-2">
                        <label htmlFor={field.key} className="block text-sm font-medium text-gray-500">
                          {formatFieldName(field.key)}
                        </label>
                        <div className="block w-full rounded-md border border-gray-200 bg-gray-50 py-2 px-3 text-sm text-gray-700">
                          {typeof field.value === 'object' ? JSON.stringify(field.value) : 
                           field.value === '' ? '-' : String(field.value)}
                        </div>
                      </div>
                    ))
                  }
                </div>
              </div>
              
              
              <div className="flex justify-end pt-4">
                <button
                  type="submit"
                  className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50"
                  disabled={saving}
                >
                  {saving ? 'Saving...' : recordId ? 'Update' : 'Create'}
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </div>
  );
}

// Export the main component wrapped in Suspense
export default function UpdateRecordPage() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="bg-white shadow rounded-lg p-6">
              <h1 className="text-2xl font-bold mb-6">Update Record</h1>
              <p>Loading update form...</p>
            </div>
          </div>
        </div>
      </div>
    }>
      <UpdateRecordContent />
    </Suspense>
  );
}
