'use client';

import { Fragment, useState, useEffect, Suspense, useRef } from 'react';
import Link from 'next/link';
import { useSearchParams } from 'next/navigation';
import { useDatabaseContext } from '@/lib/context/DatabaseContext';
import { trackPadDisplayMobile, trackPadDisplayShort } from '@/lib/config/field-names';
import NavBar from '@/components/NavBar';

interface Record {
  [key: string]: any;
}

// Main content component that uses useSearchParams
function TasksContent() {
  const searchParams = useSearchParams();
  
  // State for client 
  const [clientId, setClientId] = useState<string | null>(null);
  const [clientName, setClientName] = useState<string>('');

  // Get client ID and name from URL param or session storage - MUST run before any fetch
  useEffect(() => {
    // Check URL parameters first
    const urlClientId = searchParams.get('clid');
    if (urlClientId) {
      console.log(`Client ID found in URL: ${urlClientId}`);
      setClientId(urlClientId);
      // Store in session for persistence
      sessionStorage.setItem('currentClientId', urlClientId);
      
      // Get client name from session storage
      const storedClientName = sessionStorage.getItem('cl-name');
      if (storedClientName) {
        console.log(`Client name found in session: ${storedClientName}`);
        setClientName(storedClientName);
      }
      
      // CRITICAL: Set query params immediately with client ID filter
      setQueryParams(`$filter=Name_id eq ${urlClientId}`);
      return;
    }
    
    // Fall back to session storage
    const storedClientId = sessionStorage.getItem('currentClientId');
    if (storedClientId) {
      console.log(`Client ID found in session: ${storedClientId}`);
      setClientId(storedClientId);
      
      // Get client name from session storage
      const storedClientName = sessionStorage.getItem('cl-name');
      if (storedClientName) {
        console.log(`Client name found in session: ${storedClientName}`);
        setClientName(storedClientName);
      }
      
      // CRITICAL: Set query params immediately with client ID filter
      setQueryParams(`$filter=Name_id eq ${storedClientId}`);
    }
  }, []);  // Empty dependency array to ensure this runs once before anything else
  
  const { 
    currentHostId, 
    currentDatabaseName, 
    currentTableName,
    currentTable,
    setCurrentTableName
  } = useDatabaseContext();
  
  // Combined effect to set table name only once
  useEffect(() => {
    // Force table to be 'TrackPad' for this specialized view
    console.log('Setting table to TrackPad. Current host/db:', { currentHostId, currentDatabaseName });
    setCurrentTableName('TrackPad');
  }, [setCurrentTableName, currentHostId, currentDatabaseName]);
  
  // Debug current context values
  useEffect(() => {
    console.log('Current database context values:', {
      host: currentHostId,
      database: currentDatabaseName,
      table: currentTableName,
      tableConfig: currentTable
    });
  }, [currentHostId, currentDatabaseName, currentTableName, currentTable]);
  
  const [records, setRecords] = useState<Record[]>([]);
  const [sortedRecords, setSortedRecords] = useState<Record[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [queryParams, setQueryParams] = useState('');
  const [queryInput, setQueryInput] = useState('');
  const [searchField, setSearchField] = useState('Message');
  const [searchOperation, setSearchOperation] = useState('contains');
  
  // Sorting state - default to StartDate descending (newest first)
  const [sortField, setSortField] = useState<string>('StartDate');
  const [sortDirection, setSortDirection] = useState<'asc' | 'desc'>('desc');
  
  // Calculate number of rows needed for message textarea (1-4 rows)
  const calculateMessageRows = (text: string): number => {
    if (!text) return 1;
    const lineCount = text.split('\n').length;
    return Math.min(Math.max(1, lineCount), 4);
  };
  
  // Format date for display and ensure proper sorting
  const formatDate = (dateStr: string): string => {
    if (!dateStr) return '';
    try {
      const date = new Date(dateStr);
      if (isNaN(date.getTime())) return dateStr; // Return original if invalid
      return date.toLocaleDateString('en-US', { 
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        timeZone: 'UTC' // Ensure consistent timezone handling
      });
    } catch (e) {
      return dateStr; // Return original if parsing fails
    }
  };
  
  // Refresh trigger for manual refreshes
  const [refreshTrigger, setRefreshTrigger] = useState<number>(0);
  
  // Add custom scrollbar styles on client side only
  useEffect(() => {
    const styleElement = document.createElement('style');
    styleElement.textContent = `
      .custom-scrollbar::-webkit-scrollbar {
        width: 8px;
        height: 8px;
      }
      .custom-scrollbar::-webkit-scrollbar-track {
        background: #f7fafc;
      }
      .custom-scrollbar::-webkit-scrollbar-thumb {
        background-color: #cbd5e0;
        border-radius: 4px;
      }
      .custom-scrollbar::-webkit-scrollbar-thumb:hover {
        background-color: #a0aec0;
      }
    `;
    
    // Add the style element to the document head
    document.head.appendChild(styleElement);
    
    // Clean up function to remove the style element when component unmounts
    return () => {
      document.head.removeChild(styleElement);
    };
  }, []);

  // Get the fields from the current table configuration
  const tableFields = currentTable?.fields || [];
  
  // Check if mobile view
  const [isMobile, setIsMobile] = useState(false);
  
  // Update mobile state on window resize
  useEffect(() => {
    const checkIfMobile = () => {
      setIsMobile(window.innerWidth <= 768);
    };
    
    // Initial check
    checkIfMobile();
    
    // Add event listener for window resize
    window.addEventListener('resize', checkIfMobile);
    
    // Cleanup
    return () => window.removeEventListener('resize', checkIfMobile);
  }, []);
  
  // Mobile field labels mapping
  const mobileLabels: {[key: string]: string} = {
    'NoteType': 'Type', 
    'StartDate': 'Date',
    'pl_Status': 'Status',
    'Message': 'm',
    'pl_Cat': 'Cat',
    'Name': 'Name',
    'z_recid': 'ID'
  };
  
  // Mobile field order - using configuration from field-names.ts
  const mobileFields = [...trackPadDisplayMobile];
  
  // Get visible columns based on mobile/desktop view
  const getVisibleColumns = (): string[] => {
    // For mobile, use mobileFields (from trackPadDisplayMobile)
    // For desktop, use trackPadDisplayShort
    const fieldsToUse = isMobile ? mobileFields : trackPadDisplayShort;
    
    // Filter to only include fields that exist in the table
    return fieldsToUse.filter(field => 
      tableFields.some((f: { fieldName: string }) => f.fieldName === field)
    );
  };
  
  const visibleColumns = getVisibleColumns();
  
  // Format field name for display (with mobile-specific formatting)
  const formatFieldName = (field: string): string => {
    // Mobile-specific field name formatting
    if (isMobile && mobileLabels[field]) {
      return mobileLabels[field];
    }
    
    // Default formatting for both mobile and desktop
    return field
      .replace(/([A-Z])/g, ' $1')
      .replace(/^./, (str: string) => str.toUpperCase())
      .trim();
  };
  
  // Define the preferred column order based on the current table config and view
  const getPreferredOrder = (): string[] => {
    if (isMobile) {
      const mobileFields = ['Name', 'StartDate', 'pl_Status', 'Message', 'z_recid'];
      const allFields = tableFields.map((field: { fieldName: string }) => field.fieldName);
      return mobileFields.filter(field => allFields.includes(field));
    }
    
    return tableFields
      .filter((field: { hidden?: boolean }) => !field.hidden)
      .map((field: { fieldName: string }) => field.fieldName);
  };
  
  const preferredOrder = currentTable?.preferredColumnOrder || getPreferredOrder();

  // This effect has been moved to the combined effect above
  
  // Handle sorting of columns
  const handleSort = (field: string) => {
    // If clicking the same field, toggle direction
    if (sortField === field) {
      setSortDirection(sortDirection === 'asc' ? 'desc' : 'asc');
    } else {
      // If clicking a new field, set it as sort field with ascending direction
      setSortField(field);
      setSortDirection('asc');
    }
  };
  
  // Apply client-side sorting when sort parameters change
  useEffect(() => {
    if (!sortField || records.length === 0) {
      // If no sort field is specified, use the original records
      setSortedRecords([...records]);
      return;
    }
    
    // Create a copy of records to sort
    const recordsCopy = [...records];
    
    // Sort the records based on the selected field and direction
    recordsCopy.sort((a, b) => {
      // Get values, handling undefined/null values
      const aValue = a[sortField] || '';
      const bValue = b[sortField] || '';
      
      // Handle different data types
      if (sortField === 'StartDate') {
        // Ensure proper date parsing by standardizing the format
        // This addresses potential timezone issues with date strings
        const parseDate = (dateStr: any): number => {
          if (!dateStr) return 0;
          try {
            // For YYYY-MM-DD format, ensure it's treated as UTC
            if (typeof dateStr === 'string' && dateStr.match(/^\d{4}-\d{2}-\d{2}$/)) {
              // Parse as UTC to avoid timezone issues
              const [year, month, day] = dateStr.split('-').map(Number);
              return new Date(Date.UTC(year, month - 1, day)).getTime();
            }
            return new Date(dateStr).getTime();
          } catch (e) {
            return 0;
          }
        };
        
        const dateA = parseDate(aValue);
        const dateB = parseDate(bValue);
        
        // Debug date sorting
        console.log(`Comparing dates: ${aValue} (${new Date(dateA).toISOString()}) vs ${bValue} (${new Date(dateB).toISOString()})`);
        console.log(`Sort direction: ${sortDirection}, Result: ${sortDirection === 'asc' ? dateA - dateB : dateB - dateA}`);
        
        return sortDirection === 'asc' ? dateA - dateB : dateB - dateA;
      } else {
        // String comparison (case-insensitive)
        const strA = String(aValue).toLowerCase();
        const strB = String(bValue).toLowerCase();
        
        if (sortDirection === 'asc') {
          return strA.localeCompare(strB);
        } else {
          return strB.localeCompare(strA);
        }
      }
    });
    
    // Update the sorted records
    setSortedRecords(recordsCopy);
  }, [records, sortField, sortDirection]);
  
  // Set up initial sort parameters - but don't trigger a fetch without client ID
  useEffect(() => {
    // Skip if we don't have the necessary context
    if (!currentHostId || !currentDatabaseName || !currentTableName) {
      return;
    }
    
    // Set default sort parameters - we'll handle combined sorting in the fetch
    setSortField('StartDate');
    setSortDirection('desc');
  }, [currentHostId, currentDatabaseName, currentTableName]);
  
  // Fetch records when database selection or query params change
  useEffect(() => {
    let isMounted = true;
    
    async function fetchRecords() {
      // Don't fetch if we're still initializing
      if (!currentHostId || !currentDatabaseName || !currentTableName) {
        console.log('Waiting for initialization...', { 
          currentHostId, 
          currentDatabaseName, 
          currentTableName 
        });
        return;
      }
      
      // CRITICAL: Don't fetch if we don't have a client ID from URL
      const urlClientId = searchParams.get('clid');
      if (!clientId && urlClientId) {
        console.log('Waiting for client ID to be set from URL parameter...');
        return;
      }
      
      // Log client ID for debugging
      if (clientId) {
        console.log(`Fetching with client ID: ${clientId}`);
      }
      
      console.log('Starting to fetch records with:', {
        host: currentHostId,
        db: currentDatabaseName,
        table: currentTableName,
        queryParams
      });
      
      setLoading(true);
      setError(null);
      
      try {
        // Ensure database and table names are properly encoded for URL
        const encodedDbName = encodeURIComponent(currentDatabaseName);
        const encodedTableName = encodeURIComponent(currentTableName);
        
        // Build the base URL without query parameters
        const baseUrl = `/api/odata/${encodedDbName}/${encodedTableName}`;
        
        // Construct the URL with query parameters
        let urlWithParams = baseUrl;
        
        // Build query parameters
        let finalQueryParams = '';
        
        // Start with existing query params if any
        if (queryParams) {
          finalQueryParams = queryParams.startsWith('?') ? queryParams.substring(1) : queryParams;
        }
        
        // CRITICAL: Always ensure client ID filter is included if available
        if (clientId) {
          const clientFilter = `Name_id eq ${clientId}`;
          
          if (finalQueryParams) {
            // If we already have query params, check if we need to add the client filter
            if (finalQueryParams.includes('$filter=')) {
              // If there's already a filter, check if it includes our client filter
              if (!finalQueryParams.includes(clientFilter)) {
                // Add client filter to existing filter
                finalQueryParams = finalQueryParams.replace('$filter=', `$filter=(`) + `) and ${clientFilter}`;
              }
            } else {
              // No filter yet, add it
              finalQueryParams += `&$filter=${clientFilter}`;
            }
          } else {
            // No params yet, just add the client filter
            finalQueryParams = `$filter=${clientFilter}`;
          }
          
          console.log(`Applied client ID filter: ${clientFilter}. Final params: ${finalQueryParams}`);
        }
        
        // Add sorting if specified
        if (sortField) {
          // Sort by both StartDate and starttime24_c with newest date and time at top
          const sortParam = `$orderby=StartDate ${sortDirection},starttime24_c ${sortDirection}`;
          
          // Append or add the sort parameter
          if (finalQueryParams) {
            // Check if finalQueryParams already has $orderby
            if (finalQueryParams.includes('$orderby=')) {
              // Replace existing $orderby parameter
              finalQueryParams = finalQueryParams.replace(/\$orderby=[^&]+/, sortParam);
            } else {
              finalQueryParams += `&${sortParam}`;
            }
          } else {
            finalQueryParams = sortParam;
          }
        }
        
        // Apply the final query parameters
        if (finalQueryParams) {
          urlWithParams = `${baseUrl}?${finalQueryParams}`;
        }
        
        // Add timestamp for cache busting
        const appendChar = urlWithParams.includes('?') ? '&' : '?';
        urlWithParams = `${urlWithParams}${appendChar}_t=${new Date().getTime().toString()}`;
        
        console.log('Query params:', queryParams);
        console.log('Final URL with params:', urlWithParams);
        
        console.log('Fetching records from:', urlWithParams);
        
        const response = await fetch(urlWithParams, {
          headers: {
            'X-Host-Id': currentHostId,
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'OData-Version': '4.0',
            'OData-MaxVersion': '4.0',
            'Cache-Control': 'no-cache, no-store, must-revalidate',
            'Pragma': 'no-cache',
            'Expires': '0'
          },
          cache: 'no-store'
        });
        
        if (!isMounted) return;
        
        if (!response.ok) {
          const errorText = await response.text();
          console.error('Error response:', errorText);
          throw new Error(`Error fetching records: ${response.status} - ${response.statusText}`);
        }
        
        const data = await response.json();
        
        if (!isMounted) return;
        
        console.log('Raw API response type:', typeof data);
        console.log('Raw API response keys:', data ? Object.keys(data) : 'no data');
        console.log('Raw API response:', data);
        
        // Handle OData response format
        let recordsData = [];
        if (Array.isArray(data)) {
          console.log('Data is an array with length:', data.length);
          recordsData = data;
        } else if (data && typeof data === 'object') {
          // Handle OData response with value property
          console.log('Data is an object. Has value property:', data.hasOwnProperty('value'));
          if (data.hasOwnProperty('value')) {
            console.log('Value property type:', typeof data.value);
            console.log('Value property is array:', Array.isArray(data.value));
            console.log('Value property length:', Array.isArray(data.value) ? data.value.length : 'not an array');
          }
          recordsData = data.value || [];
        }
        
        console.log('Processed records data:', {
          isArray: Array.isArray(recordsData),
          recordCount: recordsData.length,
          firstRecord: recordsData[0] || 'no records'
        });
        
        // If we still have no records, try to extract data differently
        if (recordsData.length === 0 && data && typeof data === 'object') {
          console.log('Attempting alternative data extraction methods');
          // Look for any array property that might contain records
          for (const key in data) {
            if (Array.isArray(data[key]) && data[key].length > 0) {
              console.log(`Found potential records in property: ${key} with ${data[key].length} items`);
              recordsData = data[key];
              break;
            }
          }
        }
        
        // Log the entire data response JSON
        console.log('FULL DATA RESPONSE JSON:');
        // Log each record individually to avoid truncation
        if (data && data.value && Array.isArray(data.value)) {
          console.log(`Total records: ${data.value.length}`);
          data.value.forEach((record: any, index: number) => {
            console.log(`Record ${index + 1}:`, record);
            // Log specific fields that might be of interest
            if (record) {
              console.log(`ID: ${record.id || 'N/A'}, Name: ${record.Name || 'N/A'}, StartDate: ${record.StartDate || 'N/A'}`);
            }
          });
        } else {
          console.log('Raw data:', data);
        }
        
        // Also add the data to window for inspection
        if (typeof window !== 'undefined') {
          (window as any).taskData = data;
          console.log('Data attached to window.taskData - inspect in console');
          
          // Debug date formats in the data
          if (data && data.value && Array.isArray(data.value)) {
            console.log('DATE FORMAT ANALYSIS:');
            const dateMap = new Map();
            data.value.forEach((record: any) => {
              if (record && record.StartDate) {
                const dateStr = record.StartDate;
                dateMap.set(dateStr, (dateMap.get(dateStr) || 0) + 1);
              }
            });
            
            // Log unique dates and their counts
            console.log('Unique dates in data:');
            dateMap.forEach((count, dateStr) => {
              const parsedDate = new Date(dateStr);
              console.log(`Date string: ${dateStr}, Parsed: ${parsedDate.toISOString()}, Count: ${count}`);
            });
          }
        }
        
        // Set the records in state
        if (isMounted) {
          setRecords(recordsData);
          setLoading(false);
        }
      } catch (err: any) {
        console.error('Error in fetchRecords:', err);
        if (isMounted) {
          setError(err.message || 'Failed to fetch records');
        }
      } finally {
        if (isMounted) {
          setLoading(false);
        }
      }
    }

    fetchRecords();
    
    return () => {
      isMounted = false;
    };
  }, [currentHostId, currentDatabaseName, currentTableName, queryParams, refreshTrigger]);

  // Get hidden fields from the table configuration
  const hiddenFields = currentTable?.hiddenFields || [];
  
  // Extract column names from records and table configuration for display
  const tableColumns = (() => {
    // If displayShort is defined in the table configuration, use it
    if (currentTable?.displayShort && currentTable.displayShort.length > 0) {
      return currentTable.displayShort;
    }
    
    // Fallback to default fields if displayShort is not available
    return [
      'z_recid',
      'Name',
      'StartDate',
      'StartTime',
      'pl_Status',
      'pl_Cat',
      'Message'
    ];
  })();

  // Format cell value for display based on field type
  const formatCellValue = (field: string, value: any): string => {
    if (value === null || value === undefined) return '-';
    
    // Handle Message field special formatting
    if (field === 'Message' && typeof value === 'string') {
      // Replace newline characters with ' - ' and trim any extra whitespace
      const formattedValue = value.replace(/[\r\n]+/g, ' - ').trim();
      // Truncate to 15 characters with ellipsis if needed
      return formattedValue.length > 15 ? `${formattedValue.substring(0, 15)}...` : formattedValue;
    }
    
    // Get field configuration if available
    const fieldConfig = currentTable?.fields?.find(f => f.fieldName === field);
    
    if (fieldConfig) {
      // Format based on field type
      switch (fieldConfig.type) {
        case 'date':
          try {
            // Special handling for StartDate field
            if (field === 'StartDate') {
              return formatDate(value);
            }
            const date = new Date(value);
            return date.toLocaleDateString();
          } catch (e) {
            return String(value);
          }
        case 'boolean':
          return value ? 'Yes' : 'No';
        case 'number':
          // Check if this is a record ID field
          if (field === 'z_recid' || field.toLowerCase().includes('id')) {
            return String(value); // Display IDs as simple numbers without formatting
          }
          return typeof value === 'number' ? value.toLocaleString() : String(value);
        default:
          if (typeof value === 'object') return JSON.stringify(value);
          return String(value);
      }
    }
    
    // Default formatting if no field config
    if (typeof value === 'object') return JSON.stringify(value);
    return String(value)
  };

  // Extract record ID from @id field or other potential ID fields
  const getRecordId = (record: Record): string => {
    // Debug log the record to see available fields
    console.log('Getting ID for record:', record);
    
    // Check @odata.id field first (standard OData field)
    if (record['@odata.id']) {
      console.log('Found @odata.id:', record['@odata.id']);
      const match = record['@odata.id'].match(/\((\d+)\)/);
      if (match && match[1]) {
        console.log('Extracted ID from @odata.id:', match[1]);
        return match[1];
      }
    }
    
    // Check @id field (some OData implementations use this)
    if (record['@id']) {
      console.log('Found @id:', record['@id']);
      const match = record['@id'].match(/\((\d+)\)/);
      if (match && match[1]) {
        console.log('Extracted ID from @id:', match[1]);
        return match[1];
      }
    }
    
    // Fallback to other ID fields if URL extraction fails
    const possibleIdFields = [
      // Common ID field names
      'ID', 'Id', 'id', 'z_recid', 'recid', '_id',
      // FileMaker specific
      'RecordID', 'record_id', 'Nid', 'nid', 'ctid',
      // Other common patterns
      'key', 'primaryKey', 'recordId', 'recordKey'
    ];
    
    // Try exact matches first
    for (const idField of possibleIdFields) {
      if (record[idField] !== undefined && record[idField] !== null) {
        console.log(`Found ID in field '${idField}':`, record[idField]);
        return String(record[idField]);
      }
    }
    
    // Log available fields for debugging
    console.log('Available fields in record:', Object.keys(record));
    
    // If no ID is found, try to find any field that might contain an ID
    for (const [key, value] of Object.entries(record)) {
      // Skip metadata fields and complex objects
      if (key.startsWith('@') || key.startsWith('_') || typeof value === 'object') {
        continue;
      }
      
      // If the field name suggests it might be an ID
      const lowerKey = key.toLowerCase();
      if ((lowerKey.includes('id') || lowerKey.includes('rec') || lowerKey.endsWith('_id')) && 
          value !== null && value !== undefined) {
        console.log(`Using fallback ID field '${key}':`, value);
        return String(value);
      }
    }
    
    console.warn('Could not determine record ID for record:', record);
    return '';
  };

  // Handle query input change
  const handleQueryInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setQueryInput(e.target.value);
  };

  // Handle search field change
  const handleSearchFieldChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearchField(e.target.value);
  };

  // Handle search operation change
  const handleSearchOperationChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearchOperation(e.target.value);
  };
  
  // Handle form submission for query parameters
  const handleSubmitQuery = (e: React.FormEvent) => {
    e.preventDefault();
    
    // Get the search value from input
    const searchValue = queryInput.trim();
    
    console.log('Search submitted:', { searchField, searchValue, searchOperation, clientId });
    
    if (!searchValue) {
      // If no search value, just use client filter if available
      if (clientId) {
        setQueryParams(`$filter=Name_id eq ${clientId}`);
      } else {
        setQueryParams('');
      }
      return;
    }
    
    // If we have a specific field and operation selected, use them to build a filter
    if (searchField && searchOperation) {
      let filterExpression = '';
      
      // Special handling for date fields
      if (searchField === 'StartDate') {
        // Check for advanced date search patterns
        const dateValue = searchValue.trim();
        
        // Check for date range pattern: "from MM/DD/YYYY to MM/DD/YYYY"
        const rangeMatch = dateValue.match(/from\s+(\d{1,2}\/\d{1,2}\/\d{4})\s+to\s+(\d{1,2}\/\d{1,2}\/\d{4})/);
        if (rangeMatch) {
          const fromDate = new Date(rangeMatch[1]);
          // Add one day to the end date to make the range inclusive
          const toDate = new Date(rangeMatch[2]);
          toDate.setDate(toDate.getDate() + 1);
          
          // Format dates in ISO format for OData
          const fromIsoDate = fromDate.toISOString().split('T')[0];
          const toIsoDate = toDate.toISOString().split('T')[0];
          
          // Build range filter: StartDate ge 2023-01-01 and StartDate lt 2023-01-02
          const dateFilter = `${searchField} ge ${fromIsoDate} and ${searchField} lt ${toIsoDate}`;
          
          // Add client filter if clientId exists
          if (clientId) {
            filterExpression = `$filter=(${dateFilter}) and Name_id eq ${clientId}`;
          } else {
            filterExpression = `$filter=${dateFilter}`;
          }
        }
        // Check for greater than pattern: ">MM/DD/YYYY"
        else if (dateValue.startsWith('>')) {
          const dateStr = dateValue.substring(1).trim();
          const date = new Date(dateStr);
          const isoDate = date.toISOString().split('T')[0];
          
          if (clientId) {
            filterExpression = `$filter=${searchField} gt ${isoDate} and Name_id eq ${clientId}`;
          } else {
            filterExpression = `$filter=${searchField} gt ${isoDate}`;
          }
        }
        // Check for less than pattern: "<MM/DD/YYYY"
        else if (dateValue.startsWith('<')) {
          const dateStr = dateValue.substring(1).trim();
          const date = new Date(dateStr);
          const isoDate = date.toISOString().split('T')[0];
          
          if (clientId) {
            filterExpression = `$filter=${searchField} lt ${isoDate} and Name_id eq ${clientId}`;
          } else {
            filterExpression = `$filter=${searchField} lt ${isoDate}`;
          }
        }
        // Regular date equality
        else if (/^\d{1,2}\/\d{1,2}\/\d{4}$/.test(dateValue)) {
          const date = new Date(dateValue);
          // Add one day to handle FileMaker's date handling
          const nextDay = new Date(date);
          nextDay.setDate(nextDay.getDate() + 1);
          
          const startIsoDate = date.toISOString().split('T')[0];
          const endIsoDate = nextDay.toISOString().split('T')[0];
          
          // Use a range to get exact date: StartDate ge 2023-01-01 and StartDate lt 2023-01-02
          let dateFilter = `${searchField} ge ${startIsoDate} and ${searchField} lt ${endIsoDate}`;
          
          if (clientId) {
            filterExpression = `$filter=(${dateFilter}) and Name_id eq ${clientId}`;
          } else {
            filterExpression = `$filter=${dateFilter}`;
          }
        }
        // FileMaker OData API expects dates without quotes in filter expressions
        else {
          // Invalid date format, use as-is (will likely return no results)
          if (clientId) {
            filterExpression = `$filter=${searchField} ${searchOperation} '${searchValue}' and Name_id eq ${clientId}`;
          } else {
            filterExpression = `$filter=${searchField} ${searchOperation} '${searchValue}'`;
          }
        }
      } else {
        // For non-date fields, use the standard filter expression
        // Always treat as string for Message field to handle special characters
        const isMessageField = searchField === 'Message';
        const isNumeric = !isMessageField && !isNaN(Number(searchValue)) && !isNaN(parseFloat(searchValue));
        const formattedValue = isNumeric ? searchValue : `'${searchValue.replace(/'/g, "''")}'`;
        
        console.log('Building filter:', { searchField, searchValue, formattedValue, searchOperation });
        
        // Build the filter expression based on the search operation
        // Always use tolower() for case-insensitive searches on all fields
        if (searchOperation === 'contains') {
          const containsExpr = `contains(tolower(${searchField}), tolower(${formattedValue}))`;
            
          if (clientId) {
            filterExpression = `$filter=${containsExpr} and Name_id eq ${clientId}`;
          } else {
            filterExpression = `$filter=${containsExpr}`;
          }
        } else if (searchOperation === 'equals') {
          const equalsExpr = `tolower(${searchField}) eq tolower(${formattedValue})`;
            
          if (clientId) {
            filterExpression = `$filter=${equalsExpr} and Name_id eq ${clientId}`;
          } else {
            filterExpression = `$filter=${equalsExpr}`;
          }
        } else {
          if (clientId) {
            filterExpression = `$filter=${searchField} ${searchOperation} ${formattedValue} and Name_id eq ${clientId}`;
          } else {
            filterExpression = `$filter=${searchField} ${searchOperation} ${formattedValue}`;
          }
        }
      }
      
      console.log('Generated filter expression:', filterExpression);
      setQueryParams(filterExpression);
      return;
    }
    
    // Fallback to the original query processing if no specific field/operation is used
    const formattedValue = searchValue.startsWith('\'') && searchValue.endsWith('\'') ? 
      searchValue : `'${searchValue}'`;
    
    // Build the OData query based on the selected operation
    let processedInput = '';
    
    if (searchOperation === 'contains') {
      // Format as: contains(tolower(FieldName), tolower('value')) for case-insensitive search
      processedInput = `$filter=contains(tolower(${searchField}), tolower(${formattedValue}))`;
    } else if (searchOperation === 'equals') {
      // Format as: tolower(FieldName) eq tolower('value') for case-insensitive search
      processedInput = `$filter=tolower(${searchField}) eq tolower(${formattedValue})`;
    }
    
    console.log('Submitting query:', processedInput);
    setQueryParams(processedInput);
  };

  // Handle refresh button click while preserving client ID
  const handleRefresh = () => {
    setQueryInput('');
    // Reset to just the client ID filter if available
    if (clientId) {
      setQueryParams(`$filter=Name_id eq ${clientId}`);
    } else {
      setQueryParams('');
    }
    setSortField('StartDate');
    setSortDirection('desc');
    // Re-trigger the effect by creating a new timestamp
    setRefreshTrigger(Date.now());
  }

  // Show loading state while initializing or loading data
  if (!currentHostId || !currentDatabaseName || !currentTableName) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="flex flex-col items-center justify-center h-64 space-y-4">
              <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
              <p className="text-gray-600">Initializing database connection...</p>
              <div className="text-sm text-gray-500 bg-gray-50 p-2 rounded">
                <p>Host: {currentHostId || 'Loading...'}</p>
                <p>Database: {currentDatabaseName || 'Loading...'}</p>
                <p>Table: {currentTableName || 'Loading...'}</p>
              </div>
            </div>
          </div>
        </div>
      </div>
    );
  }

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-100">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="flex flex-col items-center justify-center h-64 space-y-4">
              <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
              <p className="text-gray-600">Loading records...</p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-100">
      <NavBar />
      
      {/* Back to Clients link */}
      <div className="max-w-7xl mx-auto pt-4 pb-0 sm:px-6 lg:px-8">
        <Link 
          href="/clients-all"
          className="inline-flex items-center px-4 py-2 mb-4 bg-blue-600 text-white rounded-md hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
        >
          <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5 mr-2" viewBox="0 0 20 20" fill="currentColor">
            <path fillRule="evenodd" d="M9.707 16.707a1 1 0 01-1.414 0l-6-6a1 1 0 010-1.414l6-6a1 1 0 011.414 1.414L5.414 9H17a1 1 0 110 2H5.414l4.293 4.293a1 1 0 010 1.414z" clipRule="evenodd" />
          </svg>
          Back to All Clients
        </Link>
      </div>
      
      {/* Client information display from session storage */}
      {(clientName || clientId) && (
        <div className="max-w-7xl mx-auto pb-0 sm:px-6 lg:px-8">
          <div className="bg-blue-50 rounded-lg shadow-md p-4 mb-4">
            <div className="text-center">
              <h2 className="text-2xl font-bold text-blue-800">
                Client: {clientName || 'Unknown'}
              </h2>
              <p className="text-lg text-blue-600">
                Client ID: {clientId || 'Not specified'}
              </p>
            </div>
          </div>
        </div>
      )}
      
      <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
        <div className="bg-white rounded-lg shadow-lg overflow-hidden">
          <div className="px-3 sm:px-6 py-4 border-b border-gray-200">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-3">
              <h1 className="text-xl font-semibold text-gray-900">
                Record List
              </h1>
              <div className="flex gap-2">
                <Link
                  href={clientId ? `/task-new?clid=${clientId}` : '/task-new'}
                  className="px-3 py-1 bg-blue-600 text-white rounded hover:bg-blue-700 text-sm"
                >
                  Add New
                </Link>
              </div>
            </div>
          </div>
          
          <div className="p-1 sm:p-3 md:p-4">
            <form onSubmit={handleSubmitQuery} className="flex flex-col gap-2 mb-6">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                <div>
                  <label htmlFor="searchField" className="block text-sm font-medium text-gray-700 mb-1">
                    Field
                  </label>
                  <select
                    id="searchField"
                    className="w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    value={searchField}
                    onChange={(e) => setSearchField(e.target.value)}
                  >
                    <option value="Message">Message</option>
                    <option value="pl_Status">Status</option>
                    <option value="pl_Cat">Category</option>
                    <option value="StartDate">StartDate</option>
                  </select>
                </div>
                
                <div>
                  <label htmlFor="queryInput" className="block text-sm font-medium text-gray-700 mb-1">
                    Search Term
                  </label>
                  <input
                    type="text"
                    id="queryInput"
                    className="w-full rounded-md border border-gray-300 py-2 px-3 text-sm shadow-sm focus:border-blue-500 focus:outline-none focus:ring-1 focus:ring-blue-500"
                    value={queryInput}
                    onChange={handleQueryInputChange}
                    placeholder={searchField === 'StartDate' ? "Enter date (e.g., 6/7/2025, >6/7/2025)" : "Enter search term (e.g., Dave)"}
                  />
                  {searchField === 'StartDate' && (
                    <p className="mt-1 text-xs text-gray-500">
                      Formats: MM/DD/YYYY, &gt;MM/DD/YYYY, &lt;MM/DD/YYYY, or from MM/DD/YYYY to MM/DD/YYYY
                    </p>
                  )}
                </div>
                
                <div>
                  <fieldset className="mt-2">
                    <legend className="block text-sm font-medium text-gray-700 mb-1">Search Type</legend>
                    <div className="flex items-center space-x-6">
                      <div className="flex items-center">
                        <input
                          id="contains"
                          name="searchOperation"
                          type="radio"
                          value="contains"
                          checked={searchOperation === 'contains'}
                          onChange={handleSearchOperationChange}
                          className="h-4 w-4 border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <label htmlFor="contains" className="ml-2 block text-sm text-gray-700">
                          Contains
                        </label>
                      </div>
                      <div className="flex items-center">
                        <input
                          id="equals"
                          name="searchOperation"
                          type="radio"
                          value="equals"
                          checked={searchOperation === 'equals'}
                          onChange={handleSearchOperationChange}
                          className="h-4 w-4 border-gray-300 text-blue-600 focus:ring-blue-500"
                        />
                        <label htmlFor="equals" className="ml-2 block text-sm text-gray-700">
                          Equals
                        </label>
                      </div>
                    </div>
                  </fieldset>
                </div>
              </div>
              
              <div className="flex justify-end mt-2">
                <button
                  type="submit"
                  className="w-full sm:w-auto px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-opacity-50"
                  disabled={loading}
                >
                  {loading ? 'Loading...' : 'Search'}
                </button>
                <button
                  type="button"
                  onClick={handleRefresh}
                  className="w-full sm:w-auto ml-2 px-4 py-2 bg-gray-200 text-gray-700 rounded hover:bg-gray-300 focus:outline-none focus:ring-2 focus:ring-gray-400 focus:ring-opacity-50"
                  disabled={loading}
                >
                  {loading ? 'Loading...' : 'Refresh'}
                </button>
              </div>
            </form>
            
            {error && (
              <div className="bg-red-50 border-l-4 border-red-400 p-4 mb-6">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-red-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-red-700">{error}</p>
                  </div>
                </div>
              </div>
            )}
            
            {!loading && !error && records.length === 0 && (
              <div className="bg-yellow-50 border-l-4 border-yellow-400 p-4 mb-6">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <svg className="h-5 w-5 text-yellow-400" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20" fill="currentColor">
                      <path fillRule="evenodd" d="M8.485 2.495c.673-1.167 2.357-1.167 3.03 0l6.28 10.875c.673 1.167-.17 2.625-1.516 2.625H3.72c-1.347 0-2.189-1.458-1.515-2.625L8.485 2.495zM10 5a.75.75 0 01.75.75v3.5a.75.75 0 01-1.5 0v-3.5A.75.75 0 0110 5zm0 9a1 1 0 100-2 1 1 0 000 2z" clipRule="evenodd" />
                    </svg>
                  </div>
                  <div className="ml-3">
                    <p className="text-sm text-yellow-700">No records found.</p>
                  </div>
                </div>
              </div>
            )}
            
            {!loading && !error && records.length > 0 && (
              <div className="relative">
                <div className="overflow-x-auto w-full custom-scrollbar">
                  <div className="min-w-max w-full">
                    <table className="min-w-full divide-y divide-gray-300">
                  <thead>
                    <tr>
                      <th className="sticky left-0 z-10 px-2 py-3 text-left text-sm font-semibold text-gray-900 bg-gray-100 w-auto">
                        <span className="whitespace-nowrap">Actions</span>
                      </th>
                      {(isMobile ? mobileFields.filter(field => field !== 'Message') : visibleColumns).map((column: string) => {
                        // Check if this column should be sortable
                        const isSortable = ['Name', 'StartDate', 'pl_Status', 'pl_Cat'].includes(column);
                        const displayName = column === 'StartDate' ? 'Start Date' : 
                                          column === 'pl_Status' ? 'Status' : 
                                          column === 'pl_Cat' ? 'Category' : 
                                          formatFieldName(column);
                        
                        // Determine if this column is the current sort field
                        const isCurrentSortField = sortField === column;
                        
                        return (
                          <th
                            key={column}
                            className={`px-3 py-3.5 text-left text-sm font-semibold text-gray-900 bg-gray-100 max-w-[200px] ${
                              isSortable ? 'cursor-pointer hover:bg-gray-200' : ''
                            }`}
                            onClick={() => isSortable && handleSort(column)}
                          >
                            <div className="flex items-center justify-between">
                              <span>{displayName}</span>
                              {isSortable && isCurrentSortField && (
                                <span className="ml-1">
                                  {sortDirection === 'asc' ? '↑' : '↓'}
                                </span>
                              )}
                            </div>
                          </th>
                        );
                      })}
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-200 bg-white">
                    {sortedRecords.map((record, index) => {
                      // Ensure recordId is always a string
                      const recordId = String(getRecordId(record) || index);
                      const bgColor = index % 2 === 0 ? 'bg-white' : 'bg-gray-50';
                      // Ensure tt_id is a string if it exists
                      const ttId = record.tt_id ? String(record.tt_id) : null;
                      
                      if (isMobile) {
                        // For mobile, use mobileFields and filter out Message for the first row
                        const mainColumns = mobileFields.filter(col => col !== 'Message');
                        const hasMessage = mobileFields.includes('Message');
                        
                        return (
                          <Fragment key={`fragment-${recordId}`}>
                            {/* Main row with action buttons and non-message fields */}
                            <tr className={bgColor}>
                              <td className="sticky left-0 z-10 px-2 py-3 text-gray-800 w-auto bg-white">
                                <div className="flex flex-row flex-nowrap gap-1 justify-start">
                                  <Link
                                    href={`/task?tid=${recordId}${clientId ? `&clid=${clientId}` : ''}`}
                                    className="px-2 py-1 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-xs"
                                    onClick={() => {
                                      if (typeof window !== 'undefined') {
                                        sessionStorage.setItem('tsk-recid', recordId);
                                        if (record.tt_id) {
                                          sessionStorage.setItem('tt_id', record.tt_id);
                                        }
                                      }
                                    }}
                                  >
                                    View
                                  </Link>
                                  <Link
                                    href={`/task-update?tid=${recordId}${clientId ? `&clid=${clientId}` : ''}`}
                                    className="px-2 py-1 bg-blue-100 text-blue-700 rounded hover:bg-blue-200 text-xs"
                                    onClick={() => {
                                      if (typeof window !== 'undefined') {
                                        sessionStorage.setItem('tsk-recid', recordId);
                                        if (record.tt_id) {
                                          sessionStorage.setItem('tt_id', record.tt_id);
                                        }
                                      }
                                    }}
                                  >
                                    Edit
                                  </Link>
                                </div>
                              </td>
                              {mainColumns.map((column: string, idx: number) => {
                                // Use tt_id if available, otherwise fall back to index
                                const cellKey = record.tt_id ? `${record.tt_id}-${column}` : `${recordId}-${column}-${idx}`;
                                return (
                                  <td
                                    key={cellKey}
                                    className="whitespace-nowrap px-2 py-3 text-gray-800 text-sm"
                                  >
                                    {formatCellValue(column, record[column])}
                                  </td>
                                );
                              })}
                            </tr>
                            
                            {/* Separate row for Message field */}
                            {hasMessage && (
                              <tr key={`${recordId}-message`} className={bgColor}>
                                <td colSpan={visibleColumns.length + 1} className="px-2 py-2 text-gray-800 text-sm border-t border-gray-200">
                                  <textarea 
                                    key={`${recordId}-textarea`}
                                    readOnly
                                    rows={calculateMessageRows(record['Message'] || '')}
                                    className="w-full p-2 text-lg border border-gray-300 rounded resize-none focus:outline-none focus:ring-1 focus:ring-blue-500"
                                    style={{ 
                                      fontSize: '20px',
                                      lineHeight: '1.4',
                                      minHeight: '2.5rem',
                                      maxHeight: '8.5rem',
                                      overflowY: 'hidden'
                                    }}
                                    value={record['Message'] || ''}
                                  />
                                </td>
                              </tr>
                            )}
                          </Fragment>
                        );
                      }
                      
                      // Desktop view - single row layout
                      return (
                        <tr key={recordId} className={bgColor}>
                          <td className="sticky left-0 z-10 px-2 py-3 text-gray-800 w-auto bg-white">
                            <div className="flex flex-row flex-nowrap gap-1 justify-start">
                              <Link
                                href={`/task?tid=${recordId}${clientId ? `&clid=${clientId}` : ''}`}
                                className="px-2 py-1 bg-gray-100 text-gray-700 rounded hover:bg-gray-200 text-xs"
                                onClick={() => {
                                  if (typeof window !== 'undefined') {
                                    sessionStorage.setItem('tsk-recid', recordId);
                                    if (record.tt_id) {
                                      sessionStorage.setItem('tt_id', record.tt_id);
                                    }
                                  }
                                }}
                              >
                                View
                              </Link>
                              <Link
                                href={`/task-update?tid=${recordId}${clientId ? `&clid=${clientId}` : ''}`}
                                className="px-2 py-1 bg-blue-100 text-blue-700 rounded hover:bg-blue-200 text-xs"
                                onClick={() => {
                                  if (typeof window !== 'undefined') {
                                    sessionStorage.setItem('tsk-recid', recordId);
                                    if (record.tt_id) {
                                      sessionStorage.setItem('tt_id', record.tt_id);
                                    }
                                  }
                                }}
                              >
                                Edit
                              </Link>
                            </div>
                          </td>
                          {visibleColumns.map((column: string) => (
                            <td
                              key={column}
                              className={`whitespace-nowrap px-2 py-3 text-gray-800 ${column.toString() === 'Message' ? 'max-w-[180px] truncate' : ''}`}
                            >
                              {formatCellValue(column, record[column])}
                            </td>
                          ))}
                        </tr>
                      );
                    })}
                  </tbody>
                    </table>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}

// Export the main component wrapped in Suspense
export default function ListPage() {
  return (
    <Suspense fallback={
      <div className="min-h-screen bg-gray-50">
        <NavBar />
        <div className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
          <div className="px-4 py-6 sm:px-0">
            <div className="bg-white shadow rounded-lg p-6">
              <h1 className="text-2xl font-bold mb-6">Tasks</h1>
              <p>Loading tasks...</p>
            </div>
          </div>
        </div>
      </div>
    }>
      <TasksContent />
    </Suspense>
  );
}
